import os
from dotenv import load_dotenv
from cav_integrated import PanelManager
from bs4 import BeautifulSoup

def get_all_service_names():
    # Looks for SERVICE1_NAME, SERVICE2_NAME, ... up to 5
    names = []
    for i in range(1, 6):
        name = os.getenv(f"SERVICE{i}_NAME")
        if name:
            names.append(name)
    return names

def extract_packages_for_panel(panel, panel_name):
    results = []
    # Authenticate
    auth_result = panel.authenticate()
    if not auth_result.get("success"):
        results.append(f"  [!] Authentication failed: {auth_result.get('message')}")
        return results

    # Fetch the Add User page
    add_user_url = f"{panel.panel_url}/user_reseller.php"
    try:
        resp = panel.session.get(add_user_url, timeout=10)
        if resp.status_code != 200:
            results.append(f"  [!] Failed to load Add User page: {resp.status_code}")
            return results
        soup = BeautifulSoup(resp.text, "html.parser")
        select = soup.find("select", {"name": "bouquet[]"}) or soup.find("select", {"name": "package"})
        if not select:
            results.append("  [!] Could not find package/bouquet dropdown in Add User page.")
            return results
        options = select.find_all("option")
        if not options:
            results.append("  [!] No package/bouquet options found.")
            return results
        for opt in options:
            if not opt.get("value") or opt.get("value") in ("", "0"):
                continue
            line = f"{opt['value']}: {opt.text.strip()}"
            results.append(line)
    except Exception as e:
        results.append(f"  [!] Error scraping Add User page: {e}")
    return results

def main():
    load_dotenv()
    output_file = "all_panel_packages.txt"
    panel_manager = PanelManager()
    service_names = get_all_service_names()
    if not service_names:
        print("No SERVICE*_NAME variables found in .env.")
        return

    with open(output_file, "w", encoding="utf-8") as f:
        for service_name in service_names:
            f.write(f"=== {service_name} ===\n")
            print(f"\n=== {service_name} ===")
            panel = panel_manager.get_panel_for_service(service_name)
            if not panel:
                msg = f"  [!] Could not get panel for service: {service_name}"
                print(msg)
                f.write(msg + "\n\n")
                continue
            results = extract_packages_for_panel(panel, service_name)
            for line in results:
                print(line)
                f.write(line + "\n")
            f.write("\n")
    print(f"\nDone! All package lists written to {output_file}")

if __name__ == "__main__":
    main()